<?php

namespace WP_Statistics\Service\Admin;

class AddOnDecorator
{
    private $addOn;
    private $isActivated = false;
    private $status;

    /**
     * @param $addOn
     */
    public function __construct($addOn)
    {
        $this->addOn = $addOn;
    }

    public function getName()
    {
        return $this->addOn->name;
    }

    public function getSlug()
    {
        return $this->addOn->slug;
    }

    public function getUrl()
    {
        return $this->addOn->url . '?' . AddOnsFactory::$addOnUtm[$this->getSlug()];
    }

    public function getDescription()
    {
        return $this->addOn->description;
    }

    public function getIcon()
    {
        return $this->addOn->icon;
    }

    public function getImage()
    {
        return $this->addOn->image;
    }

    public function getPrice()
    {
        return $this->addOn->price;
    }

    public function getVersion()
    {
        return $this->addOn->version;
    }

    public function isFeatured()
    {
        return $this->addOn->is_feature == true ? true : false;
    }

    public function getFeaturedLabel()
    {
        return $this->addOn->featured_label;
    }

    public function isExist()
    {
        return file_exists(WP_PLUGIN_DIR . '/' . $this->getPluginName());
    }

    public function isEnabled()
    {
        return is_plugin_active($this->getPluginName());
    }

    public function isActivated()
    {
        $this->status = $this->getRemoteStatus();

        return $this->isActivated;
    }

    public function getOptionName()
    {
        return AddOnsFactory::getSettingNameByKey($this->getSlug());
    }

    public function getLicense()
    {
        $option = get_option($this->getOptionName());
        return isset($option['license_key']) ? $option['license_key'] : '';
    }

    public function getPluginName()
    {
        return sprintf('%s/%s.php', $this->getSlug(), $this->getSlug());
    }

    public function getActivateUrl()
    {
        return add_query_arg([
            'action'   => 'activate',
            'plugin'   => $this->getPluginName(),
            '_wpnonce' => wp_create_nonce("activate-plugin_{$this->getPluginName()}")
        ], admin_url('plugins.php'));
    }

    public function getDeactivateUrl()
    {
        return add_query_arg([
            'action'   => 'deactivate',
            'plugin'   => $this->getPluginName(),
            '_wpnonce' => wp_create_nonce("deactivate-plugin_{$this->getPluginName()}")
        ], admin_url('plugins.php'));
    }

    public function getStatus()
    {
        if ($this->isEnabled()) {

            $this->status = $this->getRemoteStatus();

            if (is_wp_error($this->status)) {
                $this->updateStatuses(false);
                return $this->status->get_error_message();
            }

            if ($this->status) {
                $this->updateStatuses(true);
                return __('Activated', 'wp-statistics');
            } else {
                $this->updateStatuses(false);
                return __('Not activated', 'wp-statistics');
            }

        } else if ($this->isExist()) {
            $this->updateStatuses(false);
            return __('Inactive', 'wp-statistics');
        }

        return __('Not installed', 'wp-statistics');
    }

    public function getRemoteStatus()
    {
        // Avoid remote request
        if (!$this->isExist() or !$this->getLicense()) {
            return false;
        }

        // Cache the status
        if ($this->status) {
            return $this->status;
        }

        $transientKey         = AddOnsFactory::getLicenseTransientKey($this->getSlug());
        $downloadTransientKey = AddOnsFactory::getDownloadTransientKey($this->getSlug());

        // Get any existing copy of our transient data
        if (false === ($response = get_transient($transientKey))) {
            $args = add_query_arg([
                'plugin-name' => $this->getSlug(),
                'license_key' => $this->getLicense(),
                'website'     => get_bloginfo('url'),
            ], WP_STATISTICS_SITE . '/wp-json/plugins/v1/validate');

            $response = wp_remote_get($args, [
                'timeout' => 35,
            ]);

            if (is_wp_error($response)) {
                return $response;
            }

            $body     = wp_remote_retrieve_body($response);
            $response = json_decode($body, false);

            set_transient($transientKey, $response, DAY_IN_SECONDS);
        }

        if (isset($response->code) && $response->code == 'error') {
            return new \WP_Error($response->data->status, $response->message);
        }

        if (isset($response->status) and $response->status == 200) {
            $this->isActivated = true;

            // To clear the download transient and sync with download status
            delete_transient($downloadTransientKey);

            return true;
        }
    }

    private function updateStatuses($status)
    {
        $statues                     = get_option('wp_statistics_activate_addons', []);
        $statues[$this->addOn->slug] = $status;

        unset($statues['add-ons-bundle']);

        update_option('wp_statistics_activate_addons', $statues);
    }

    public static function countActivatedAddOns()
    {
        return array_sum(get_option('wp_statistics_activate_addons', []));
    }
}